/*
    COPYTABL.PRG
    05/07/1998
    Ken Mayer
    BorCon 98
    A program designed to copy a table structure and it's
    index tags without using the old xBase DML
    commands. (Special thanks to Ken Chan for helping
    me get parts of this working ...)

    Problem: The UpdateSet object, when it copies
    a table does not copy the index tags.

    Solution: Use a combination of TableDef, local
    SQL, and database.createIndex().

    Of course, one might ask, why do all that,
    when other methods of copying tables exist?
       COPY TABLE (local SQL) actually works pretty well,
         but it's a bit tricky copying from one
         type of table to another.
       database.copyTable() will only copy
         from one table type to the same table type.

    In addition, this is being used as an example
    of various ways to use some of the undocumented
    features of Visual dBASE 7, and I had to have
    an excuse, eh?

    -----------------------------------------------------
    WARNING: I haven't tried every single field type
     or permutation possible with this program -- if it
     doesn't work for your situation examine the code
     which I have attempted to document as well as I can ...
     there may still be a way to make it work - if you find
     it, let me know, and I'll try to fix for a future 
     version.
    -----------------------------------------------------

    USAGE:
       do CopyTabl with "<cOldTable>", "<cNewTable>" ;
                        ["<cOldDatabase>" [, "<cNewDatabase>" ]]
          where: cOldTable   = table being copied
                 cNewTable   = table being created
                 cOldDatabase = database alias (BDE)  ** Optional
                 cNewDatabase = same for table being created ** Optional
          the database parameters are optional, but are necessary
          if a) you want to copy a table from a SQL server or
                use a table from a BDE alias (i.e., :MUGS: ...); and
             b) you want to create the new table on a SQL server or
                create a table in a BDE alias (i.e., :MUGS: ...)

       NOTE: If you use a BDE alias and a .DBF (or .DB) for
       the table being copied, you MUST use the extension for
       the table name (i.e., INVENTORY.DBF) - if you are using
       purely local tables, the extension is not necessary.

       EXAMPLE:  
         do CopyTabl with "INVENTORY.DBF", "TestIt", "MUGS"
           Will copy the Inventory table in the MUGS alias to
           the local directory as "TestIt"
         do CopyTabl with "INVENTORY.DBF", "TestIt.DBF", "MUGS", "MUGS"
           Will copy the MUGS inventory table to MUGS testit table
         do CopyTabl with "FISH","Test.DB"
           Will copy the FISH.DBF table to TEST.DB (Paradox table format)

       =====================================================
       WARNING: If the "new table" already exists, this 
       will erase it without any warning to you ...
       =====================================================

       MORE INFORMATION THAN YOU MAY WANT TO KNOW:
         1) If you are copying a .DBF that does not have
            a primary key, to a .DB, which requires a primary
            key in order to create any other indexes,
            no index tags will be copied -- there is no way
            for a program to know which field should be the
            primary key. 
         2) If you are copying a table with no indexes,
            this will still work ...
         3) If you copy a table with a primary key, the primary
            key will be duplicated. However, note that
            if the primary key field is not the first field
            in the table, it will be in the copy -- this is
            because .DB tables require that the primary key
            be the first field in the table, and it was easier
            to just make sure that the primary key was always
            first in the copy, then spend more time trying
            to determine if the table was a .DB or a .DBF ...

   ------------------------------------------------------------------
   Now on to the code:
   ------------------------------------------------------------------
*/

parameters cOldTable, cNewTable, cOldDatabase, cNewDatabase

// ---------------------------------------------------
// add BDE Aliases/Database references here if needed
// for the copy table ...:
cCopyOldTable = iif( not empty( cOldDatabase ), ;
                    ":"+cOldDatabase+":"+cOldTable,;
                    cOldTable )
cCopyNewTable = iif( not empty( cNewDatabase ), ;
                    ":"+cNewDatabase+":"+cNewTable,;
                    cNewTable )

// ------------------------------------
// Need to actually open the databases:
if not empty( cOldDatabase )
   d = new Database()
   d.databaseName := cOldDatabase
   d.active       := true
endif
if not empty( cNewDatabase )
   d2 = new Database()
   d2.databasename := cNewDatabase
   d2.active       := true
endif

// ---------------------------------------
// Destroy table if it already exists ...:
if _app.databases[1].tableExists( "&cCopyNewTable." )
   drop table "&cCopyNewTable."
endif

// ------------------------------------------
// First, copy the table structure:
// Create an instance of the tableDef object:
t = new TableDef()

// ------------------------------------------
// Next, if we're using a database for the 
// first table, we need to establish it ...
if not empty( cOldDatabase )
   t.database     := d
endif

// -------------
// assign table
t.tableName := cOldTable
// Load it ...
t.Load()
// ------------------------------------
cPrimary = t.primaryKey  // primary key

/*
   -------------------------------------------------
   Due to the fact that a .DB table must
   have the primary key first, we are going
   to load a second array that will contain
   the field names, types, decimal and decimalLength
   properties of the fields in the table being
   copied. This will have the primary key
   (if any) as the first field, and all other
   fields will come after in order ...
   -------------------------------------------------
*/
aFields = new Array( t.fields.size, 4 )
// Get the first field:
if not empty( cPrimary )
   aFields[ 1, 1 ] = cPrimary
   // unfortunately, most of the standard
   // array methods don't exist here, so can't just
   // use scan():
   for i = 1 to t.fields.size
       if t.fields[ i ].fieldName == cPrimary
          nPrimary = i
          exit
       endif
   next
else
   // no primary key:
   nPrimary = 1
   aFields[ 1, 1 ] = t.fields[ nPrimary ].fieldName
endif
aFields[ 1, 2 ] = t.fields[ nPrimary ].type
aFields[ 1, 3 ] = t.fields[ nPrimary ].length
aFields[ 1, 4 ] = t.fields[ nPrimary ].decimalLength

// ---------------------------------------------------
// load the rest of the fields skipping the primary 
// (or first field) which we already have ...
nFields = 0
for i = 2 to ( t.fields.size )
    nFields++
    // Skip primary key OR if no primary, skip first field
    if ( empty( cPrimary ) and nFields == 1 );
       OR t.fields[ nFields ].fieldName == cPrimary
       nFields++
    endif
    aFields[ i, 1 ] = t.fields[ nFields ].fieldName
    aFields[ i, 2 ] = t.fields[ nFields ].type
    aFields[ i, 3 ] = t.fields[ nFields ].length
    aFields[ i, 4 ] = t.fields[ nFields ].decimalLength
next

// ------------------------------------------
// if new table type is .DB
lDB = upper( right( cNewTable, 3 ) ) == ".DB"

// --------------------------------------------
// Now we loop through the fields, building the
// CREATE TABLE command:
cCreate = [CREATE TABLE "]+cCopyNewTable+[" (]

// ---------------------------------------------
// Here's the loop!
for nField = 1 to t.fields.size
    // must have a comma separator between
    // fields (but don't put one in until after
    // we have a field ...)
    if nField > 1
       cCreate += ", "
    endif

    // Add fieldname
    cCreate +=         [x."]+aFields[ nField, 1 ] + [" ]

    // get other properties we need:
    nLength        = aFields[ nField, 3 ]
    nDecimalLength = iif( lDB, 0, aFields[ nField, 4 ] )

    // -------------------------------------------------
    // deal with type
    // As per notes in the header, if there is a problem
    // with a specific field type not being created
    // properly, it will be in the following CASE
    // statement:
    cType          = aFields[ nField, 2 ]
    do case
       case upper(cType) $ "SHORT,LONG"
            cCreate += "SMALLINT"
       case upper(cType) $ "LONG INTEGER"
            cCreate += "INTEGER"
       case upper(cType) $ "BCD,NUMERIC"
            if lDB
               cCreate += "NUMERIC("+nLength+")"
            else
               cCreate += "NUMERIC("+nLength+","+nDecimalLength+")"
            endif
       case upper(cType) $ "NUMBER,DOUBLE"
            cCreate += "FLOAT("+nLength+","+nDecimalLength+")"
       case upper(cType) $ "ALPHA,CHARACTER"
            cCreate += "CHARACTER("+nLength+")"
       case upper(cType) $ "VARCHAR"
            cCreate += "VARCHAR("+nLength+")"
       case upper(cType) $ "DATE"
            cCreate += "DATE"
       case upper(cType) $ "LOGICAL"
            cCreate += "BOOLEAN"
       case upper(cType) $ "MEMO"
            cCreate += "BLOB(0,1)"
       case upper(cType) $ "BINARY"
            cCreate += "BLOB(0,2)"
       case upper(cType) $ "FORMATTED MEMO" // .DB, not .DBF
            if lDB
               cCreate += "BLOB(0,3)"
            else
               cCreate += "BLOB(0,1)"  // regular memo?
            endif
       case upper(cType) $ "OLE"
            cCreate += "BLOB(0,4)"
       case upper(cType) $ "GRAPHIC"        // .DB, not .DBF
            if lDB
               cCreate += "BLOB(0,5)"
            else
               cCreate += "BLOB(0,2)"  // Binary
            endif
       case upper(cType) $ "TIME"           // .DB, not .DBF
            if lDB
               cCreate += "TIME"
            else
               cCreate += "TIMESTAMP"      // ?
            endif
       case upper(cType) $ "TIMESTAMP"
            cCreate += "TIMESTAMP"
       case upper(cType) $ "MONEY"         // .DB
            if lDB
               cCreate += "MONEY"
            else
               cCreate += "NUMERIC(20,4)"
            endif
       case upper(cType) $ "AUTOINCREMENT"
            cCreate += "AUTOINC"
       case upper(cType) $ "BYTES"         // .DB
            if lDB
               cCreate += "BYTES("+nLength+")"
            else
               cCreate += "SMALLINT"
            endif
    endcase
next
if not empty( cPrimary ) // primary key
   cCreate += [, PRIMARY KEY( x."]+cPrimary+[")]
endif
cCreate += [)]

// Execute create command
//? cCreate // show SQL command generated
&cCreate.

// -------------------------------------------
// Now we need a second tableDef for the 
// second table, to determine THAT table type:
t2 = new TableDef()
if type( "d2" ) # "U"
   t2.database  := d2
endif
t2.tableName := cNewTable
t2.Load()
cNewTableType = t2.tableType

lIndexes = false // flag for msgbox call later
/*
   -----------------------------------------------------
   IF 1) the table being created is a .DB AND there's a
         primary key

         OR

         it's not a .DB being created

         AND 

      2) There are any indexes in the first place ...

   THEN we want to copy the indexes ...
   -----------------------------------------------------
*/
if ( (lDB AND not empty( cPrimary ) );
   OR NOT lDB );
   AND t.indexes.size > 0

   // flag to let user know ...:
   lIndexes = true

   // if we're working off a .DBF, some of the 
   // index properties are different:
   lDBF = ( t.indexes[ 1 ].className == "DBFINDEX" )

   // Loop through the index array:
   for nTag = 1 to t.indexes.size

       // if the second table is DBF, we need
       // to set properties:
       if cNewTableType == "DBASE"
          i = new DBFINDEX()
          if lDBF // table being copied from is DBASE
             i.forExpression := t.indexes[ nTag ].forExpression
             i.Expression    := t.indexes[ nTag ].expression
             i.Type          := t.indexes[ nTag ].type
          else  // table being copied from is NOT dBASE
             i.Expression    := t.indexes[ nTag ].fields
          endif
       else
          i = new INDEX()
          if lDBF // table being copied from is DBASE
             i.Fields        := t.indexes[ nTag ].expression
          else
             i.Fields        := t.indexes[ nTag ].fields
             i.caseSensitive := t.indexes[ nTag ].caseSensitive
          endif // lDBF
       endif  // cNewTableType == "DBASE"

       // Shared properties
       i.descending       := t.indexes[ nTag ].descending
       i.indexName        := t.indexes[ nTag ].indexName
       i.unique           := t.indexes[ nTag ].unique

       // Here's where we create the index ...
       if type( "d2" ) # "U"
          d2.createIndex( cNewTable, i )
       else
          _app.Databases[1].createIndex( cCopyNewTable, i )
       endif
   next
endif

// Let 'em know we're finished:
cIndex = iif( lIndexes, " and indexes were copied",;
                        "but no indexes were copied" )
msgbox( "Table structure of ["+cCopyOldtable+;
        "] has been copied to ["+;
         cCopyNewTable+"], "+cIndex+".", "Done!" )

close database

/*
    -------------------------------------------------------
    End of Program: COPYTABL.PRG
    -------------------------------------------------------
*/
