// File: DBF7File.Cc

/*======================================================================

Programmer: Bowen Moursund (bowen@nerds.com)
Last modified: 08/13/1998
Written for: Visual dBASE 7.0

------------------------------------------------------------------------

Purpose:
Extends Jim Sare's BFile class with level 7 DBF utility methods.

------------------------------------------------------------------------

Notes:
The SetNextAutoIncValue method is based on information supplied by
Jay Parsons and acquired second hand via Ken Mayer <g>. Ken's notes
concerning Jay's info:

    According to Jay:
      "1. In a VdB7 .DBF, the main header is longer, 44h (68 bytes).
          The table of field descriptors starts next, at offset
          44h/68d.
       2. The field name comes first, as usual. The type byte
          indicating the type of the field is 32 bytes into the
          descriptor, which is 48 bytes long.
       3. The Autoinc type is marked by a "+" type byte, 2Bh.
       4. The next available autoinc number starts 10 bytes
          farther along and is stored in 4 bytes binary, least-
          significant first.

------------------------------------------------------------------------

Examples of use:

   set procedure to DBF7File.CC additive
   oDBF7File = new DBF7File( "Foo.DBF" )
   oDBF7File.SetNextAutoIncValue( "FOO_ID", 1 ) // reset autoinc field

   oDBF7File.SetMDXFlag( false ) // detach production MDX

======================================================================*/


class DBF7File(cDBF) of BFile from "BFile.Cc"
   set procedure to BFile.Cc additive
   protect DBF, DBFDef

   class::SetDBF(cDBF)

   function SetDBF
      parameters cDBF
      local bSuccess
      bSuccess = false

      if argCount() == 1 and type("cDBF") == "C" and file(cDBF)
         this.DBF = cDBF
         bSuccess := true
      else
         this.DBF = null
         bSuccess := false
      endif
      return bSuccess

   function GetDBF
      return this.DBF

   function VerifyDBF
      return type("this.DBF") == "C" and file(this.DBF)

   function SetNextAutoIncValue( cFieldName, nNextValue )
      if not this.VerifyDBF()
         return 1 // DBF not found
      endif
      local i, nReturn, nFieldNo, oDBFDef, oFields, bIsField, bIsAutoinc
      nReturn = 0
      nFieldNo = 0
      oDBFDef = new tableDef()
      oDBFDef.tableName = this.DBF
      try
         oDBFDef.load()
      catch (exception e)
         return 6  // Could not load tableDef object
      endtry
      oFields = oDBFDef.fields
      bIsField = false
      bIsAutoinc = false
      for i = 1 to oFields.size
         if upper(oFields[i].fieldName) == upper(cFieldName)
            bIsField := true
            if oFields[i].type == "AUTOINC"
               bIsAutoinc := true
               nFieldNo := i
            endif
            exit
         endif
      next i
      if not bIsField
         return 2 // Field name not found
      endif
      if not bIsAutoinc
         return 3 // Field not an autoinc field
      endif
      try
         this.open(this.DBF, "RW")
      catch (exception e)
         return 4 // Cannot open DBF
      endtry
      nOffset = (68+(48*(nFieldNo-1))+42)
      if this.seek( nOffSet, 0 ) <> nOffset
         this.close()
         return 5 // Low level seek failed
      endif
      this.WriteULong( nNextValue )
      this.close()
      return 0 // Success!

   function SetMDXFlag( bValue )
      if not this.VerifyDBF()
         return 1 // DBF not found
      endif
      try
         this.open(this.DBF, "RW")
      catch (exception e)
         return 4 // Cannot open DBF
      endtry
      if this.seek( 28, 0 ) <> 28
         this.close()
         return 5 // Low level seek failed
      endif
      this.write( iif(bValue, chr(1), chr(0)) )
      this.close()
      return 0 // Success!

endclass

// End of file: DBF7File.Cc
