/*
	Filename: 	sdQuery.cc
	Date:			December 2, 1998
   Author:		Gary White

	Purpose: sdQuery is a query object that supports "soft deletes".
   In addition to the normal query properties, the following methods
   are added to the ROWSET object:

	ShowDeleted( bShow )		// if bShow is true, deleted records
									// will be displayed.  If bShow is false
                           // deleted records will be hidden (default).

	RecallRecord()				// recalls a record marked as deleted.

   IsRecDeleted()				// returns TRUE if the current record is
								   // marked as deleted

	Two properties have been added to the ROWSET object:

   DeletedVisible				// is a boolean value that will tell you
									// if deleted records are visible
									// this property is read-only.  setting
									// it has no effect other that temporarily
									// causing it to have an improper value

   SoftDelete					// is a boolean value that indicates whether or 
									// or not the underlying table supports soft deletes.

	In addition, a calculated field called "Deleted" is added to the rowset fields array.
   Changing the value to true will delete the record.  Setting it to value to false will
   recall the record, i.e.

   	form.rowset.fields["Deleted"].value = True	// Delete the record
   	form.rowset.fields["Deleted"].value = False	// Recall the record

   Assigning a value other than a logical will be ignored.

   The calculated "Deleted" field may be displayed on a form, or in a browse.  Note, however,
   that the form designer will not allow you to Pick the deleted field from a field list for 
   a checkbox.  You may, however, assign the datalink manually by typing in the datalink in 
   the inspector:

   	form.rowset.fields["Deleted"]

	See the sample form "sdQuery.wfm" for an example.   	

	Usage:  
		set procedure to sdquery.cc additive
		q = new sdQuery()
      q.sql = "select * from myTable"
      q.active = true
      q.rowset.showDeleted( true )	// will display deleted records

      Alteratively:
      Design your form normally.
      Open in source editor
      Below the opening CLASS line, add:
			set procedure to sdquery.cc additive

      Then change the line that defines the query object:
			this.<somename> = new query()
      To
			this.<somename> = new sdQuery()

      If you subsequently load the form in the form designer, VdB will
      erroneously stream the line:
      	execute = class::execute

      You must manually delete this line
*/
class sdQuery of query()
	protect softDelete, deletedVisible

	this.rowset.ShowDeleted = class::ShowDeleted
   this.rowset.recallRecord = class::recallRecord
   this.rowset.isRecDeleted = class::isRecDeleted
   this.rowset.DeletedVisible = false
   this.rowset.canDelete = class::canDelete
   this.rowset.onDelete = class::onDelete

   function execute
   	// defines and protype externs used by the class
      #define curSOFTDELETEON 327689

      // define an error for table not supporting soft deletes
      e = new exception()
      e.code = 9999
      e.message = "Table does not support soft deletes"

      this.softDelete = true	// assume table supports soft deletes
   	if super::execute()
         if type( "DbiUndeleteRecord" ) # "FP"
            extern cint DbiUndeleteRecord ( CHANDLE ) IDAPI32
         endif
         if type ( "DbiGetRecord" ) # "FP"
            extern cint DbiGetRecord (CHANDLE, CINT, CPTR, CPTR) IDAPI32
         endif
         if type ( "DbiSetProp" ) # "FP"
            extern cint DbiSetProp( CHANDLE, CUINT, CLOGICAL ) IDAPI32
         endif
         if type("DbiValidateProp") # "FP"
            extern CINT DbiValidateProp( CHANDLE, CLONG, CLOGICAL ) idapi32
         endif
         if DbiValidateProp( this.rowset.handle, curSOFTDELETEON, True ) # 0
            msgbox( "Table does not support soft deletes", "Alert", 48 )
            this.softDelete := false
         endif
         if this.softDelete
         	// create a calculated field to hold the Deleted value for the row
         	if type("this.rowset.fields['Deleted']") == "U"
            	f = new field()
               f.fieldName = "Deleted"
               this.rowset.fields.add( f )
               f.beforeGetValue = {||this.parent.parent.isRecDeleted()}
               f.canChange = {|newValue| type("newValue")=="L"}
               f.onChange = {;if this.parent.parent.isRecDeleted();this.parent.parent.recallRecord();else;this.parent.parent.delete();endif}
            endif
         endif
      endif
		return true

   function ShowDeleted( bDeletedOn )
   	// sets the rowset to display or hide deleted records depending on the
      // value of the bDeletedOn paramater.
      // if bDeletedOn = true, deleted records will be displayed
      // if bDeletedOff = true, deleted records will be hidden (VdB default)
      // Return value: TRUE if successful / FALSE if it fails
   	local n
      if bDeletedOn == this.rowset.deletedVisible	// no change necessary
      	return true
      endif
      n = -1
      if ( this.active ) and ( this.softDelete )
         n = DbiSetProp( this.rowset.handle, curSOFTDELETEON, bDeletedOn )
         if n # 0
            msgbox( "BDE Error - Unable to set property", "Error", 16 )
         else
            this.rowset.deletedVisible = bDeletedOn
         endif
         this.rowset.refreshControls()
      endif
      return ( n == 0 )

   function recallRecord
   	// recalls a record marked for deletion.
      // Return value: TRUE if successful / FALSE if it fails
   	local n, b
      n = -1
      if ( this.active ) and ( this.softDelete )
         b = this.rowset.bookmark()
         if this.rowset.isRecDeleted()
            n = DbiUndeleteRecord( this.rowset.handle )
            if n # 0
               MsgBox( "Recall failed", "Error", 16 )
            endif
            this.rowset.goto( b )
         else
            MsgBox( "Record is not deleted", "Alert", 48 )
         endif
      endif
      return ( n == 0 )

   function isRecDeleted
   	// returns true if the current record is marked deleted
      // false if the record is not marked deleted
      // null if the rowset does not support soft deletes, or the
      // query is not active
   	local IsDel
      IsDel = null
      if ( this.active ) and ( this.softDelete )
         c = space( 8 )
         if DbiGetRecord( this.rowset.handle, 0, "", c ) == 0
            IsDel := asc( substr( c, 7, 1 ) ) # 0
         endif
      endif
      return IsDel

	function canDelete
   	if type("this.deletedVisible") == "L" and this.deletedVisible
      	this.savePosition = this.bookmark()
      endif
      return true

	function onDelete
   	if type("this.deletedVisible") == "L" and this.deletedVisible
      	if type("this.savePosition") # "U"
      		this.goto( this.savePosition )
         endif
      endif
      return

endclass

