/* 
   ----------------------------------------------------------------
   FormCntl.CC
   Authors.: Ken Mayer
             Gary White
   Date....: 10/26/1998

   This file contains custom control definitions for each
   control I use in a system. The idea is to derive all controls
   from these, so that a simple change here will update everything
   in the system. 

   You should check the headers for each control -- in some
   cases, there are very explicit instructions (the custom 
   date entry fields and custom grids in particular).

   This version of the file includes a lot more class definitions
   than earlier versions, and if you examine each, most of the 
   properties are included, so that if you wish to make a change, 
   the property is already here, even if it's set to the default. 

   WARNING: the kmCustDateEntryField and kmCustDateEntryField2
   classes use HOLIDAY.CC, which in turn uses HEBREW.CC and 
   DATEEX.CC, as well as the dUFLP.H file, and MAKEHOLS.DBF. 
   If any of these are not deployed and you use this 
   entryfield (which is rather handy, if I may say so myself 
   <G>) then it won't work ... see HOLIDAYS.TXT for details
   on what needs to be deployed -- there's quite a lot.

   Special thanks to Gary White for the work on the date entryfield
   kmCustGrid and holiday classes!

   The controls here are:
   ---------------------------------------------------------------
   TEXT CONTROLS:
   kmFormTitle            -- text with a border, larger font ...
   kmVerticalFormTitle1   -- vertical title subclassed from above
                             with text rotated 90 degrees ...
   kmVerticalFormTitle2   -- same, but text rotated 270 degrees
   kmCustText             -- text control, generally used with 
                             entryfields and such
   kmOtherCustText        -- other text not used for entryfields

   DATA CONTROLS (ones that can be used to link to data):
   kmCustEntryField       -- entryfield with various properties set
   kmCustDateEntryField   -- Special date entryfield
                             used for "business" purposes
                             to move a date off a holiday or
                             weekend!
   kmCustDateEntryField2  -- Similar to above, allows USER to
                             specify which way to go with date,
                             or to leave it "as is".
   kmDisableEntryField    -- as it says -- a disabled entryfield
   kmCustSpinbox          -- as it says
   kmCustEditor           -- editor
   kmCustCombobox         -- style 2 combobox
   kmCustComboboxStyle1   -- style 1 subclass of above
   kmCustComboboxStyle0   -- style 0 subclass of kmCustCombox
   kmCustListbox          -- listbox
   kmCustCheckBox         -- checkbox
   kmCustRadio            -- radiobutton
   kmCustImage            -- some basic settings
   kmCustGrid             -- a bunch of the options get set
                             AND you can set "global" fonts/colors
                             for your grid ... see comments and code
                             for details
   kmCustEditGrid         -- grid that allows editing
   
   GADGETS:
   kmCustRectangle        -- etched-out, no text
   kmCustContainer        -- transparent, same border as rectangle
   kmCustNotebook         -- notebook ...
   kmCustTabBox           -- custom tabbox control
   kmCustSlider           -- "super class" for next two subclasses
   kmCustHorizontalSlider -- Horizontal, tics on bottom
   kmCustVerticalSlider   -- vertical, tics on both sides
   kmCustProgress         -- really basic
   kmTreeView             -- basically this is Keith Chuvala's
                             treeView from ICon 98
   kmTreeItem             -- used with above (not a custom control)
   kmTreeContainer        -- has a kmTreeView, with two pushbuttons
                             to expand/collapse the treeview

   MISC:
   Repaint()              -- Jim Sare's function with API calls
                             to repaint a form
   VerifyDateForm         -- Form used by kmCustDateEntryField2
                             above. 

   ---------------------------------------------------------------
   NOTE: You can change font, and font sizes in one place
   for most of these controls, as well as colors ... that
   is what all the #defines are below:
   ----------------------------------------------------------------
*/

// Font info:
#DEFINE CONTROLFONTNAME "Arial"         // default
#DEFINE CONTROLFONTSIZE 10              // default

// Colors:
#DEFINE HIGHLIGHT       "White/Blue"
#DEFINE NORMAL          "Black/White"
#DEFINE NORMALNOBACK    "Black"
#DEFINE DISABLED        "White/Maroon"
#DEFINE TITLECOLOR      "BLUE"
// Grid specific colors:
#DEFINE GRIDBACK        "WHITE"
#DEFINE GRIDHEADER      "BLACK/BTNFACE"
// TabBox colors:
#DEFINE TABBOXBACK      "BLUE"        // background
#DEFINE TABBOXNORMAL    "BLUE/SILVER" // current tab
// TreeView colors:
#DEFINE TREEVIEWCOLORS  "WINDOWTEXT/WINDOW" // text/background
// Slider colors:
#DEFINE SLIDERCOLORS    "BLACK/SILVER"

// Standard border for entry objects:
#DEFINE ENTRYBORDERSTYLE  7 // used in kmCustEntryField, kmCustEditor,
                            // kmCustImage, kmCustGrid, kmCustCombobox
                            // kmCustTreeView
#DEFINE CONTAINERBORDER  10 // kmCustRectangle, kmCustContainer
#DEFINE CHECKBORDER       3 // kmCustCheckBox, kmCustRectangle
#DEFINE GADGETBORDER      0 // kmCustSlider, kmCustProgress
      /* border options (so you don't have to go look 'em up):
             0 = default
             1 = raised
             2 = lowered
             3 = none
             4 = single (single-line)
             5 = double (double-line)
             6 = drop shadow
             7 = client
             8 = modal
             9 = etched in
             10 = etched out
       */

/* 
   ----------------------------------------------------------------
   Most of my forms use a title in the upper left somewhere,
   it's rather handy to subclass it this way ... 
   ----------------------------------------------------------------
*/
class kmFormTitle( oParent ) of Text( oParent ) custom
   with (this)
        alignHorizontal := 1	// Center
        alignVertical   := 1	// Middle
        anchor          := 0  // None
        border          := true
        borderStyle     := 1  // Raised
        colorNormal     := TITLECOLOR
        fontBold        := false
        fontItalic      := false
        fontName        := CONTROLFONTNAME
        fontSize        := 18 // larger font than default...
        fontStrikeout   := false
        fontUnderline   := false
        height          := 1.3182
        metric          := 0   // Chars
        prefixEnable    := false
        printable       := true
        rotate          := 0 // Normal
        text            := "kmFormTitle"
        transparent     := true
        wrap            := false
   endwith
endclass

/*
   ----------------------------------------------------------------
   Take the formTitle control above, and rotate it 90 degrees,
   and set the height/width ... and:
   ----------------------------------------------------------------
*/
class kmVerticalFormTitle1(parentObj) of kmFormTitle(parentObj) custom
   with (this)
      height := 7.7727
      width  := 6.5714
      rotate := 1	// 90 degrees
   endwith
endclass

/*
   ----------------------------------------------------------------
   Take the VerticalFormTitle1 control above, and rotate it 270 
   degrees: 
   ----------------------------------------------------------------
*/
class kmVerticalFormTitle2(parentObj) of kmVerticalFormTitle1(parentObj) custom
   with (this)
      rotate := 3	// 270 degrees
   endwith
endclass

/* 
   ----------------------------------------------------------------
   This is the standard text control I use to mark an
   entryfield etc ...
   ----------------------------------------------------------------
*/
class kmCustText( oParent ) of Text( oParent ) custom
   with (this)
        alignHorizontal := 2   // Right
        alignVertical   := 1	 // Middle
        anchor          := 0   // None
        border          := false
        borderStyle     := 0   // Default
        colorNormal     := NORMALNOBACK
        fontBold        := false
        fontItalic      := false
        fontName        := CONTROLFONTNAME
        fontSize        := CONTROLFONTSIZE
        fontStrikeout   := false
        fontUnderline   := false
        height          := 0.9545
        metric          := 0   // chars
        prefixEnable    := true
        printable       := true
        text            := "kmCustText"
        transparent     := true
        wrap            := false
    endwith
endclass

/*
   ----------------------------------------------------------------
   Other text controls ... this is text on the left, only 
   difference, so we subclass it from above:
   ----------------------------------------------------------------
*/
class kmOtherCustText( oParent ) of kmCustText( oParent ) custom
   with (this)
        alignHorizontal := 0   // Left
   endwith
endclass

/* 
   ----------------------------------------------------------------
   Standard entryfield. The onGotFocus event can be over-ridden
   easily enough ... it's just useful to be sure that the
   cursor isn't somewhere you don't want it. I generally
   prefer selectAll to be off so that I (or a user) don't
   over-write the contents of the entryfield by accident.
   ----------------------------------------------------------------
*/
class kmCustEntryField( oParent ) of Entryfield( oParent ) custom
   with (this)
        border         := true
        borderStyle    := ENTRYBORDERSTYLE
        colorNormal    := NORMAL
        colorHighlight := HIGHLIGHT
        enabled        := true
        fontBold       := false
        fontItalic     := false
        fontName       := CONTROLFONTNAME
        fontSize       := CONTROLFONTSIZE
        fontStrikeout  := false
        fontUnderline  := false
        function       := ""
        height         := 1
        maxLength      := 25
        metric         := 0  // Chars
        mousePointer   := 0 // default
        picture        := ""
        selectAll      := false
        tabStop        := true
        value          := "kmCustEntryField"
        onGotFocus     := {; this.keyboard( "{Home}" )}
   endwith
endclass

/*
   ----------------------------------------------------------------
   This specialized entryfield uses the holidays.cc class 
   to create a holiday table, if necessary, and if not 
   necessary, check for it, and use it. It checks a date 
   to make sure it is not a holiday -- if it is, it should
   move the date OFF the holiday, either forward or
   backward (default is forward) to the next business
   day.

   This custom class HOLIDAY.CC is necessary for this
   to work properly, as is MAKEHOLS.DBF. (HOLIDAY.CC
   uses dUFLP.H, HEBREW.CC, and DATEEX.CC, which 
   also uses TIME.CC) -- This is important when you
   deploy an application -- make sure all the parts
   are there ... (see HOLIDAYS.TXT for details)

   Custom property: transfer == either "F" or any other value.
   If this value is set to "F" (default) it will move a date
   *forward* off the date, if set to anything else, it will
   move it backward (i.e., if the date is a Saturday, it 
   will move it backward to a Friday). This can be set
   in the form's onOpen, or in the control's onOpen, or if you
   wish a permament change in behavior, change it below.
   ----------------------------------------------------------------
*/
class kmCustDateEntryField( oParent ) of kmCustEntryField( oParent ) custom
   with (this)
      width = 10.8571
      value = {}   // default to a date
   endwith

   this.transfer = "F" // default

   function onLostFocus

      // if the date field is empty, don't do anything!
      if empty( this.value)
         return
      endif

      // Make sure this is open and available:
      if not setproc( "HOLIDAY.CC" )
         set procedure to holiday.cc additive
         this.setHol = true
      else
         this.setHol = false
      endif
      this.holiday = new Holiday()

      // here's the work -- run routine to 
      // check the date and move it if needed:
      this.value = this.Holiday.isHoliday( this.value, ;
                                           this.Transfer )

   return

endclass

/*
   ----------------------------------------------------------------
   This is an even MORE specialized date entryfield,
   it works similar to kmCustDateEntryField, but
   instead, it uses isHol to check the date, and if
   the date falls on a holiday, it brings up a small
   form allowing the user to select whether to leave 
   the date alone, or move it forward/backward to the
   next working date available.

   This custom class HOLIDAY.CC is necessary for this
   to work properly, as is MAKEHOLS.DBF. (HOLIDAY.CC
   uses dUFLP.H, HEBREW.CC, and DATEEX.CC, which 
   also uses TIME.CC) -- This is important when you
   deploy an application -- make sure all the parts
   are there ... (see HOLIDAYS.TXT for details)
   ----------------------------------------------------------------
*/
class kmCustDateEntryField2( oParent ) of kmCustEntryField( oParent ) custom
   with (this)
      width = 10.8571
      value = {}   // default to a date
   endwith

   this.transfer = "F" // default

   function onLostFocus

      // if the date field is empty, don't do anything!
      if empty( this.value)
         return
      endif

      // Make sure this is open and available:
      if not setproc( "HOLIDAY.CC" )
         set procedure to holiday.cc additive
         this.setHol = true
      else
         this.setHol = false
      endif
      this.holiday = new Holiday()

      // Check to see if we're on a non-working day:
      if this.Holiday.isHol( this.value )
         /* 
            We know it's a non-working date. Now
            we'll bring up a form, and ask the user
            what to do. If they want to move off the
            Non-Working-Day, we check the button clicked
            and loop through until we find a work day.
         */

         // Gary White added this (and the form at the end of
         // this file):
         f=new verifydateform()
         f.parent=this
         f.readmodal()
         f.release()

         do while this.Holiday.isHol( this.value )
            if this.transfer $ "FB"
               // transfer is either forward or backward:
               if this.transfer = "F"
                  this.value++
               else
                  this.value--
               endif // this.transfer = "F"
            else // (this.transfer = "" )
               exit // out of loop because we're leaving date
                    // "as is".
            endif // this.transfer $ "FB"
         enddo // while this.Holiday.isHol()
      endif // this.Holiday.isHol()

   return

endclass

/*
   ----------------------------------------------------------------
   Disabled entryfield ... 
   I particularly like the use of the mousePointer ...
   ----------------------------------------------------------------
*/
class kmDisableEntryField( oParent ) of kmCustEntryField( oParent ) custom
   with (this)
        colorNormal    := DISABLED
        colorHighlight := DISABLED
        when           := {; return false }
        mousePointer   := 12 // No
   endwith
endclass

/*
   ----------------------------------------------------------------
   Custom Spinbox
   I'm not really doing a lot with this one, because I don't
   use spinboxes in most of my apps ...

   You could, if using these with dates, grab the onLostFocus
   method used for the kmCustDateEntryField or kmCustDateEntryField2
   and insert it into a subclass of this spinbox set for
   a date value. Just a thought.
   ----------------------------------------------------------------
*/
class kmCustSpinbox( oParent ) of SPINBOX( oParent ) custom
   with (this)
        border         := true
        borderStyle    := ENTRYBORDERSTYLE
        colorNormal    := NORMAL
        colorHighlight := HIGHLIGHT
        enabled        := true
        fontBold       := false
        fontItalic     := false
        fontName       := CONTROLFONTNAME
        fontSize       := CONTROLFONTSIZE
        fontStrikeout  := false
        fontUnderline  := false
        function       := ""
        height         := 1
        mousePointer   := 0 // default
        picture        := ""
        rangeMax       := 100
        rangeMin       := 1
        rangeRequired  := true
        selectAll      := false
        spinOnly       := false
        step           := 1.0
        tabStop        := true
        value          := 1.0
        width          := 8.00
   endwith
endclass

/*
   ----------------------------------------------------------------
   One of the frustrating things with editor controls
   is that there's no colorHighlight property. So,
   let's make it act as if there were by changing
   the colorNormal ... the popup is handy, because it
   allows editing IN the control for simple formatting.
   However, for that to be useful, the evalTags property
   needs to be set to true also.
   ----------------------------------------------------------------
*/
class kmCustEditor( oParent ) of Editor( oParent ) custom
   with (this)
        anchor         := 0 // none
        border         := true
        borderStyle    := ENTRYBORDERSTYLE
        colorNormal    := NORMAL
        cuaTab         := true
        enabled        := true
        evalTags       := true
        fontBold       := false
        fontItalic     := false
        fontName       := CONTROLFONTNAME
        fontSize       := CONTROLFONTSIZE
        fontStrikeout  := false
        fontUnderline  := false
        height         := 3
        metric         := 0 // Chars
        modify         := true
        mousePointer   := 0 // default
        popupEnable    := true        
        scrollBar      := 2 // auto
        tabStop        := true
        value          := "kmCustEditor"
        wrap           := true
        onGotFocus     := {; this.colorNormal = HIGHLIGHT }
        onLostFocus    := {; this.colorNormal = NORMAL }
   endwith
endclass

/*
   ----------------------------------------------------------------
   Custom listbox ...
   Note the use of the "ShowSelected" method - this is handy
   for listboxes where you programmatically select a value and
   it doesn't appear on the list (because it's off the bottom
   or top ...) -- this forces the currently selected item
   to the top of the listbox.
   ----------------------------------------------------------------
*/
class kmCustListbox( oParent ) of Listbox( oParent ) custom
   with (this )
      borderStyle     := ENTRYBORDERSTYLE
      colorHighLight  := HIGHLIGHT
      colorNormal     := NORMAL
      curSel          := 1
      dataSource      := "ARRAY {'kmCustListbox'}"
      enabled         := true
      fontBold        := false
      fontItalic      := false
      fontName        := CONTROLFONTNAME
      fontSize        := CONTROLFONTSIZE
      fontStrikeout   := false
      fontUnderline   := false
      height          := 4
      mousePointer    := 0 // default
      multiple        := false
      sorted          := false
      tabStop         := true
      width           := 8
   endwith
   /* ----------------------------------------------------------
      ShowSelected() -- If the currently selected element
      in a listbox is not on screen, this will force it to appear
      at the top of the listbox. Useful for the situations where
      you programmatically set the curSel property of the listbox.

       Posted by Romain Strieff in the Borland VdBASE Newsgroups,
       and updated by Jim Sare ...

       Usage: form.mylistbox.ShowSelected( nCurSel )
       // where: nCurSel is the currently selected item
      ---------------------------------------------------------- */
   FUNCTION ShowSelected
      parameter nSelection

      if type( "SendMessage" ) # "FP"
         extern cInt SendMessage(cHandle,cint, cWord, cWord) user32 ;
                         from "SendMessageA"
      endif

      #define LB_SETTOPINDEX HtoI("0197")

      //make nth item top -- numbers are 0-based (first item is zero)
      SendMessage ( this.hwnd, LB_SETTOPINDEX, (nSelection - 1),0)

endclass

/*
   ----------------------------------------------------------------
   I always use style 2 comboboxes ...  same problem with 
   colorHighlight as editors -- doesn't exist
   ----------------------------------------------------------------
*/
class kmCustCombobox( oParent ) of Combobox( oParent ) custom
   with (this)
        autoDrop       := false
        borderStyle    := ENTRYBORDERSTYLE
        colorNormal    := NORMAL
        dropDownHeight := 6.00
        dropDownWidth  := 2.00
        enabled        := true
        fontBold       := false
        fontItalic     := false
        fontName       := CONTROLFONTNAME
        fontSize       := CONTROLFONTSIZE
        fontStrikeout  := false
        fontUnderline  := false
        height         := 1.0909
        metric         := 0 // Chars
        mousePointer   := 0 // default
        sorted         := false
        style          := 2 // DropDownList
        tabStop        := true
        value          := "kmCustCombobox"
        onGotFocus     := {; this.colorNormal = HIGHLIGHT }
        onLostFocus    := {; this.colorNormal = NORMAL }
   endwith
endclass

/*
   ----------------------------------------------------------------
   For those who want a style 1 combobox, here's a subclass
   of the above:
   ----------------------------------------------------------------
*/
class kmCustComboboxStyle1( oParent ) of kmCustCombobox( oParent ) custom
   with (this)
        style          := 1 // DropDown
        value          := "kmCustComboboxStyle1"
   endwith
endclass

/*
   ----------------------------------------------------------------
   For those who want a style 0 combobox, here's a subclass
   of the above:
   ----------------------------------------------------------------
*/
class kmCustComboboxStyle0( oParent ) of kmCustCombobox( oParent ) custom
   with (this)
        style          := 0 // Simple
        value          := "kmCustComboboxStyle0"
   endwith
endclass

/*
   ----------------------------------------------------------------
   Not a lot to say about this ....
   ----------------------------------------------------------------
*/
class kmCustCheckbox( oParent ) of CheckBox( oParent ) custom
   with (this)
        borderStyle    := CHECKBORDER
        colorNormal    := NORMALNOBACK
        enabled        := true
        fontBold       := false
        fontItalic     := false
        fontName       := CONTROLFONTNAME
        fontSize       := CONTROLFONTSIZE
        fontStrikeout  := false
        fontUnderline  := false
        group          := false
        height         := 1.0909
        metric         := 0 // Chars
        mousePointer   := 0 // default
        tabStop        := true
        text           := "kmCustCheckbox"
        textLeft       := false
        transparent    := true
        value          := false
   endwith
endclass

/*
   ----------------------------------------------------------------
   Not much to say about this, either ...
   ----------------------------------------------------------------
*/
class kmCustRadio( oParent ) of RadioButton( oParent ) custom
   with (this)
        borderStyle     := CHECKBORDER
        colorNormal     := NORMALNOBACK
        enabled         := true
        fontBold        := false
        fontItalic      := false
        fontName        := CONTROLFONTNAME
        fontSize        := CONTROLFONTSIZE
        fontStrikeout   := false
        fontUnderline   := false
        group           := true
        height          := 1.0909
        metric          := 0 // Chars
        mousePointer    := 0 // default
        tabStop         := true
        text            := "kmCustRadio"
        textLeft        := false        
        transparent     := true
        value           := true
   endwith
endclass

/*
   ----------------------------------------------------------------
   a simple rectangle, but I like the etched-out appearance:
   ----------------------------------------------------------------
*/
class kmCustRectangle( oParent ) of Rectangle( oParent ) custom
   with (this)
        border          := true
        borderStyle     := CONTAINERBORDER
        colorNormal     := "" // use defaults
        fontBold        := false
        fontItalic      := false
        fontName        := ""
        fontSize        := CONTROLFONTSIZE
        fontStrikeout   := false
        fontUnderline   := false
        height          := 4
        metric          := 0  // Chars
        mousePointer    := 0  // default
        patternStyle    := 0  // solid
        text            := "" // no text -- which with
                              // this style you wouldn't get
                              // to appear anyway!
   endwith
endclass

/*
   ----------------------------------------------------------------
   A basic container class ... this uses the "containerborder"
   definition, which defaults to the etched out look. I prefer
   my containers to be 'transparent', so that I can put something
   under them (you can put a background image on a form, and may 
   not want the container to be opaque on top of it ...), or 
   use the form's color setting ...
   ----------------------------------------------------------------
*/
class kmCustContainer( oParent ) of Container( oParent ) custom
   with (this)
      anchor          := 0 // None
      borderStyle     := CONTAINERBORDER
      colorNormal     := "" // transparent  
      expandable      := true
      height          := 10.00
      mousePointer    := 0 // default
      transparent     := true
      width           := 20.00
   endwith
endclass

/*
   ----------------------------------------------------------------
   Custom notebook control ...
   Nothing real fancy here. 
   ----------------------------------------------------------------
*/
class kmCustNotebook( oParent ) of Notebook( oParent ) custom
   with (this)
      anchor          := 0 // none
      borderStyle     := 0 // default (for a notebook, this is raised)
      buttons         := false
      colorNormal     := TABBOXNORMAL
      curSel          := 1
      dataSource      := 'ARRAY {"Tab 1", "Tab 2"}'
      enabled         := true
      focus           := 0 // Normal
      fontBold        := true
      fontItalic      := false
      fontName        := CONTROLFONTNAME
      fontSize        := CONTROLFONTSIZE
      fontStrikeout   := false
      fontUnderline   := false
      height          := 4
      mousePointer    := 0 // Default
      multiple        := false
      tabStop         := true
      visualStyle     := 0 // Right Justify
      width           := 20.00
   endwith
endclass

/*
   ----------------------------------------------------------------
    Custom Tabbox -- pretty basic, we want the tabbox to
    change the form's pageNo property to whatever the
    current selection is. When it opens, we want to
    force the page number to 1 (useful since if we
    are working in the designer, and save a form,
    it saves the "current" page number out to the
    form definition ...). The onSelChange event may
    be something you will need to override ... s'okay.
    I do it all the time ...
   ----------------------------------------------------------------
*/
class kmCustTabbox(parentObj) of TABBOX(parentObj) custom
   with (this)
      anchor         := 1 // bottom
      colorNormal    := TABBOXBACK
      colorHighLight := TABBOXNORMAL
      curSel         := 1
      dataSource     := 'ARRAY {"Individual Record","Find Record"}'
      enabled        := true
      fontBold       := true
      fontItalic     := false
      fontName       := CONTROLFONTNAME
      fontSize       := CONTROLFONTSIZE
      fontStrikeout  := false
      fontUnderline  := false
      height         := 1
      metric         := 0	// Chars
      mousePointer   := 0  // default
      tabStop        := true
      width          := 40
   endwith

   function onOpen
      this.curSel := 1
      form.pageNo := 1

   function onSelChange
      form.pageNo := this.curSel
      repaint(form) // repaint this form -- API calls 
                    // by Jim Sare -- function toward end
                    // of this file
   return
endclass

/*
   ----------------------------------------------------------------
    Custom Image object -- the important aspect of this one
    is the alignment (I hate having images distorted!) ... 
   ----------------------------------------------------------------
*/
class kmCustImage(parentObj) of IMAGE(parentObj) custom
   with (this)
      alignment    := 3	// Keep Aspect Stretch
      anchor       := 0  // none
      borderStyle  := ENTRYBORDERSTYLE
      enabled      := false
      height       := 4
      metric       := 0	// Chars
      mousePointer := 0 // default
      width        := 12
   endwith
endclass

/*
   ----------------------------------------------------------------
   Custom grid -- sets a bunch of the options -- I never
   allow editing in a grid, so it's useful to just deal with
   all that here. (However, there's an "editing" grid
   after this one that inherits the same functionality of this
   one ...)

   This grid allows the developer to set custom font/color
   properties for header/editor controls. These are handled in 
   the grid's onOpen event, by looping through the columns
   array and setting values appropriately.

   VERY IMPORTANT -- YOU MUST SET THE COLUMNS PROPERTY FOR THE
   GRID -- otherwise none of this does anything a'tall. This is
   done once you have set the dataLink property. Then click
   on the COLUMNS property in the inspector -- click the "Tool"
   button (the wrench), and select the fields you want to display
   in the grid ... click "OK" and there you go -- you can now
   programmatically assign whatever you want to ...

   -------------------------------------------------------
   NOTE: One problem is that if you store your own properties 
   in the form, the form designer is going to strip them
   out (this is a known problem with the designer and
   custom properties).

   There are a couple of possible ways around it, the simplest
   is probably to set the properties in code in the form,
   so you can test them. You could then do everything in your
   power to remember to open the form in the editor before
   you edit it, and then copy the affected code into the
   clipboard or even a temporary file, and then once
   done designing the form, go back into the editor and
   paste the code back in.

   Another is to copy them out to a header file (.H -- this 
   is simply a file you include with a #INCLUDE statement)
   and then try to remember to put this back into the form's
   definition when it gets stripped out.

   If someone comes up with a better solution, we'd love
   to hear about it!

   (Of course, if you use the exact same font/color schemes
   throughout the application, just make the changes in this
   custom control ...)

   This is one possible block of code you could copy for
   use ...:

   // Properties for kmCustEditGrid1 -- use proper name
   // as it appears in form's constructor code:
   with ( this.kmCustEditGrid1 )
		fontName = "Times New Roman"    
		fontSize = 8                    
   	headingFont = "Times New Roman" 
   	headingFontSize = 12            
      colorNormal = "RED/BLACK"       
      colorHighlight = "WHITE/BLUE"   
   endwith

   If you tinker with the fontSize property, you will
   probably want to modify the cellHeight. Unfortunately,
   that's going to take some tinkering to determine the 
   best value. 

   ----------------------------------------------------------------
*/
class kmCustGrid(parentObj) of GRID(parentObj) custom
   with (this)
      allowAddRows      := false
      allowColumnMoving := false
      allowColumnSizing := false
      allowEditing      := false
      allowRowSizing    := false
      anchor            := 0 // none
      bgColor           := GRIDBACK
      borderStyle       := ENTRYBORDERSTYLE
      cellHeight        := 0.82
      cuaTab            := false
      dragScrollRate    := 300
      enabled           := true
      gridLineWidth     := 1
      hasColumnHeadings := true
      hasColumnLines    := true
      hasIndicator      := true
      hasRowLines       := true
      hasVScrollHintText := true
      height            := 7.4545
      hScrollBar        := 0	// Off
      integralHeight    := true
      lockedColumns     := 0
      metric            := 0	// Chars
      multiSelect       := false
      rowSelect         := true
      tabStop           := true
      vScrollBar        := 1 // on
      width             := 35.4286
   endwith

   // These are the "default" properties -- set
   // them here, and they will be set in the onOpen
   // event:
   this.fontName              = CONTROLFONTNAME
   this.headingFont           = CONTROLFONTNAME
   this.fontSize              = CONTROLFONTSIZE
   this.headingFontSize       = CONTROLFONTSIZE
   this.fontBold              = false // editor controls
   this.headingFontBold       = true  // heading
   this.fontItalic            = false
   this.headingFontItalic     = false
   this.fontUnderline         = false
   this.headingFontUnderline  = false
   this.fontStrikeout         = false
   this.headingFontStrikeout  = false
   this.colorNormal           = NORMAL    
   this.headingColor          = GRIDHEADER
   this.colorHighlight        = HIGHLIGHT 

   /*
       Setup:
       The use of form.inDesign comes from much agony --
       if the code below is executed in the designer,
       the designer streams out the overriden open and
       readmodal events -- this causes a serious looping
       issue -- the form never opens and VdBASE will
       eventually crash. However, if we state that in 
       design mode we DON'T want to execute this code,
       then the events are never streamed out, and
       the world is good ...
   */
   if ( not form.inDesign ) and ;
      type( "form.initGrids" ) # "FP"

      form.initGrids  = this.initGrids
      form.sOpen      = form.open
      form.sReadModal = form.readModal
      form.open       = {; form.initGrids(); form.sOpen() }
      form.readModal  = {; form.initGrids(); form.sReadModal() }
   endif

   /*
       Simple check to remind the developer that this
       grid won't work without the columns array being
       set ... it will only fire if the columns array
       is empty when you bring up the form in design mode,
       AND when you first place the grid onto the form
       (no way around that).
   */
   function onDesignOpen
      if this.columns.size == 0
         msgbox( "Columns array is empty.", "kmCustGrid error", 16 )
      endif

   /*
      Initialize the grid or grids on a form -- this
      processing is based on Jim Sare's code for
      looping through controls -- it handles multiple
      kmCustGrids, and the same on containers ...
   */
   function initGrids
      parameter oControl
      private   oRef
      local     oParent

      oParent = iif( type( "oControl" ) == "O", oControl, FORM )
      oRef    = oParent.first
      do
         if type( "oRef.first" ) # "U"
            form.initGrids( oRef )
         endif
         if type( "oRef.setProperties" ) == "FP"
            oRef.setProperties()
         endif
         oRef = oRef.before
      until oRef.Name == oParent.first.name
   // -- end of method: initGrids()

   // Here's the work-horse for handling the custom fonts/colors ...
   function setProperties
      /*
         The idea is to set some custom font/colors for
         the grid, by looping through the columns
         array and setting these properties appropriately
      */

      // if columns array has zero elements, there's
      // nothing to do here:
      if this.columns.size == 0
         return 
      endif

      // Loop through the columns array, setting
      // values for the headingControl AND the 
      // editorControl:
      for i = 1 to this.columns.size
          heading = this.columns[i].headingControl
          editor  = this.columns[i].editorControl

          /*
             we are checking to see if current control
             is using the grid/system defaults -- if it
             is, then it's ok to override them with
             whatever the developer sets as the defaults
             above. If the value is NOT one of the defaults,
             then we don't want to over-ride it, as the
             developer may have changed something in the
             designer for the form, and this would be set
             in the constructor code ...
          */
          if heading.fontName == "Arial" // system default
             heading.fontName := this.HeadingFont
          endif
          if heading.fontSize == 10 
             heading.fontSize := this.HeadingFontSize
          endif
          if heading.fontBold
             heading.fontBold := this.HeadingfontBold
          endif
          if NOT heading.fontItalic
             heading.fontItalic := this.HeadingFontItalic
          endif
          if NOT heading.fontUnderline
             heading.fontUnderline := this.HeadingFontUnderline
          endif
          if NOT heading.fontStrikeout
             heading.fontStrikeout := this.HeadingFontStrikeout
          endif
          if heading.colorNormal == "WindowText/BtnFace" or;
             heading.colorNormal == "WindowText/Window"
             heading.colorNormal := this.HeadingColor
          endif

          if this.columns[i].editorType # 2 
                            // checkbox doesn't have font 
                            // properties
             if editor.fontName == "Arial"
                editor.fontName := this.fontName
             endif
             if editor.fontSize == 10
                editor.fontSize := this.fontSize
             endif
             if NOT editor.fontBold
                editor.fontBold := this.fontBold
             endif
             if NOT editor.fontItalic
                editor.fontItalic := this.fontItalic
             endif
             if NOT editor.fontUnderline
                editor.fontUnderline := this.fontUnderline
             endif
             if NOT editor.fontStrikeout
                editor.fontStrikeout := this.fontStrikeout
             endif
          endif // checkbox

          // Colors:
          if editor.colorNormal == "WindowText/Window"
             editor.colorNormal := this.colorNormal
          endif
          // Checkbox defaults to "WindowText/Window" for 
          // highlight
          if empty( editor.colorHighlight ) or;
             ( this.columns[i].editorType == 2 AND ;
               editor.colorHighlight == "WindowText/Window" )
             editor.colorHighlight := this.colorHighlight
          endif

          // --------------------------------------------
          // NOTE: You can specify editorControl specific
          // properties as defaults for each of the 
          // specific editorControls. A place for that is
          // added below (like above, we check to see
          // if the control's property is set to the default
          // so we don't step on whatever may have been
          // set in the form designer:

          do case
             // editorType = 0 (default) or 1 (entryfield)
             case this.columns[i].editorType <= 1
                // here are the default properties you might
                // want to set globally:
                if editor.borderStyle == 3
                   // editor.borderStyle := ENTRYBORDERSTYLE
                endif
  
             // editorType = 2 (checkbox)
             case this.columns[i].editorType == 2 
                // none
       
             // editorType = 3 (spinbox)
             case this.columns[i].editorType == 3 
                if editor.step == 1
                   // editor.step := 2 // whatever
                endif
                if editor.borderStyle == 3
                   // editor.borderStyle := ENTRYBORDERSTYLE
                endif

             // editorType = 4 (combobox)
             case this.columns[i].editorType == 4
                if editor.sorted == false
                   // editor.sorted := true
                endif
                if editor.dropDownHeight == 11.32
                   // editor.dropDownHeight := 15
                endif
                if editor.dropDownWidth == 0.00
                   // editor.dropDownWidth := 0.00
                endif
                if editor.borderStyle == 3
                   // editor.borderStyle := ENTRYBORDERSTYLE
                endif
                if editor.mousePointer == 0
                   // editor.mousePointer := 13 // hand
                endif
         endcase
      next
   return 

endclass

/*
   ----------------------------------------------------------------
   Custom editing grid -- a few properties have been
   modified, but otherwise everything else is the
   same as kmCustGrid ...
   ----------------------------------------------------------------
*/
class kmCustEditGrid(parentObj) of kmCustGrid(parentObj) custom
   with (this)
      allowAddRows      := true
      allowEditing      := true
      rowSelect         := false
   endwith
endclass

/*
   ----------------------------------------------------------------
   Sliders ... kinda fun little toys, but I don't have a lot of 
   uses for 'em. Provided here to give something basic to work 
   from. (This is the super class the other two are subclassed from)
   ----------------------------------------------------------------
*/
class kmCustSlider(parentObj) of SLIDER(parentObj) custom
   with (this)
      borderStyle     := GADGETBORDER
      colorNormal     := SLIDERCOLORS
      enabled         := true
      enableSelection := false
      metric          := 0	// Chars
      mousePointer    := 0 // default
      rangeMax        := 10
      rangeMin        := 1
      startSelection  := -1
      tabStop         := true
      value           := 0
   endwith
endclass

/*
   ----------------------------------------------------------------
   subclassed from kmSlider:
   ----------------------------------------------------------------
*/
class kmCustHorizontalSlider(parentObj) of kmCustSlider(parentObj) custom
   with (this)
      height          := 1
      width           := 29.5714
      vertical        := false
   endwith

endclass

/*
   ----------------------------------------------------------------
   subclassed from kmSlider:
   ----------------------------------------------------------------
*/
class kmCustVerticalSlider(parentObj) of kmCustSlider(parentObj) custom
   with (this)
      height          := 7.0909
      width           := 4.5714
      ticsPos         := 0	// Both
      vertical        := true
   endwith
endclass

/*
   ----------------------------------------------------------------
    Progress bar -- really useful. Be nice if we could
    change the color, but it's a Windows object that's
    been surfaced, and it uses the Windows colors ...
    When using one, drop it onto a form, and set
    the rangeMin (1 is good, but may not be what
    you want); rangeMax (if doing something with a
    table use the rowset's count() ...) and 
    set the value (start at 1, and increment as you
    do whatever you are doing ... )

    NOTE: Alan Katz discovered an odd bug -- if your rangeMax
    is 100,000 or greater the calculations are wrong. Simple
    solution: assume 100%, set the rangeMax to 100 and
    when you process your data, calculate the *percentage*
    of where you are ... if working with a table, you
    would take the rowset's count() and divide that into
    the current position and multiply by 100 ... something
    like:
        fMessage.progress1.rangeMax := 100
        for i = 1 to queryName.rowset.count() 
            fMessage.progress1.value := ;
               ( i / queryName.rowset.count() ) * 100
            // do some processing
        next

   ----------------------------------------------------------------
*/
class kmCustProgress(parentObj) of PROGRESS(parentObj) custom
   with (this)
      borderStyle  := GADGETBORDER
      height       := 1
      metric       := 0	// Chars
      mousePointer := 12	// No (can't interact with it ...)
      rangeMin     := 1
      rangeMax     := 100 // default
      value        := 25  // default
      width        := 31
   endwith
endclass

/*
   ----------------------------------------------------------------
    This treeview object is one that Keith Chuvala threw together
    and demoed at ICON 98 -- this version is only slightly 
    altered by me for inclusion here, largely for readability 
    (and to place nearly all the properties here, with some
    descriptions, as this is probably the hardest control
    to work with ...) ...
    It has four methods attached to it:
       Collapse:       Collapses all expanded treeItems
       Expand:         Expands all treeItems
       FillFromArray:  Like it says -- you must define
                       the array, first ...
          Usage: form.kmTreeView1.FillFromArray( arrayreference )
       TreeFill:       Called from FillFromArray, fills treeView
                       with treeItems

    Suggestion: put on a container with two pushbuttons
    to call the Collapse and Expand methods ... (see
    kmTreeContainer below)
   ----------------------------------------------------------------
*/
class kmTreeView(parentObj) of treeview(parentObj) CUSTOM
   with (this)
      allowEditLabels  := false // allow user to edit
      allowEditTree    := false // allow user to add/delete items
      anchor           := 0     // none
      borderStyle      := ENTRYBORDERSTYLE
      checkBoxes       := false // whether each treeItem has a checkbox
      checkedImage     := ""    // image source (file, etc.) when checked
      colorNormal      := TREEVIEWCOLORS
      disablePopup     := true  // whether tree's popup is disabled
      enabled          := true
      fontBold         := false
      fontItalic       := false
      fontName         := CONTROLFONTNAME
      fontSize         := CONTROLFONTSIZE
      fontStrikeout    := false
      fontUnderline    := false
      hasButtons       := true  // whether to display +/- buttons when there are children
      hasLines         := true  // whether to display lines
      height           := 7.9545 // to start with
      image            := ""    // default image when tree doesn't have focus
      imageScaleToFont := false // whether images match scale of font
      imageSize        := 16    // horizontal size in pixels
      indent           := 19    // horizontal indent in pixels
      linesAtRoot      := true  // whether line connets tree item at root to sub-items
      metric           := 0	  // Chars
      mousePointer     := 0     // default
      selectedImage    := ""    // default image to display when selected
      showSelAlways    := true  // whether to continue highlighting selected item
                                // even if treeview doesn't have focus
      toolTips         := true  // display text as tooltips if too long to fit in window?
      trackSelect      := true  // highlight as mouse passes over ...?
      uncheckedImage   := ""    // default image when not checked
      width            := 15    // to start with
   endwith

   procedure expand
      // Loop through tree and expand all expandable items
      local tItem
      tItem = this.firstchild
      do while tItem # NULL
         tItem.expanded = true
         tItem = tItem.nextsibling
      enddo

   procedure collapse
      // Loop through tree and collapse all collapseable items
      local tItem
      tItem = this.firstchild
      do while tItem # NULL
         tItem.expanded = false
         tItem = tItem.nextsibling
      enddo

   procedure FillFromArray
      // Loads tree from an array -- uses method treeFill to do it
      parameter aArray
      if type("aArray") # "A"
         msgbox("Array required!","Can't fill TreeView")
         return
      endif
      this.treefill(this,aArray)  // Assume it's out there!
   return

   /* 
      TreeFill: A recursive function for filling a TreeView Control
      Pass an array of strings and/or arrays of strings to define
      TreeItems, e.g. {"Item1",{"SubItem1_1","SubItem1_2"},"Item2"}
   */
   Procedure TreeFill
      Parameters tContainer, ; // Object reference to treeview or treeitem
                 aItems        // Array used to construct tree items

      local i, t, tname
      private uItem

      if type("tContainer") # "O"
         msgbox("Invalid container reference","Oops!")
         return
      endif
      if type("aItems") # "A"
         msgbox("Expecting Array reference","Oops!")
         return
      endif    

      t = tContainer

      for i = 1 to aItems.Size   
         uItem = aItems[i]   // private required to get type of aItems[i]
         if type("uItem") == "C"
            tName = "treeitem" + i
            t = new kmTreeItem(tContainer,tName)
            t.text = aItems[i]
         elseif type("uItem") == "A"         
            TreeFill(t,aItems[i])
         endif
      next i
   return

endclass

/*
   ----------------------------------------------------------------
   TreeItem -- used with TreeView:
   Note that this is not a custom control ...
   ----------------------------------------------------------------
*/
class kmTreeItem(parentObj,name) of TreeItem(parentObj,name) 
   with(this)
      bold          := false // boldface?
      checked       := false // is it "checked"?
      expanded      := false // if sub-items ...
      image         := ""    // image when not selected
      selectedImage := ""    // image when selected
   endwith
endclass

/*
   --------------------------------------------------------------
   Tree Container -- Container that has a kmTreeView control
   with a pushbutton for the two treeView events to expand
   and collapse the treeview's items ...

   NOTE: To use the fill routine to fill the treeView with
   treeItems, you need to use:
      form.kmTreeContainer1.FillIt( aArray )
         (where aArray is the array reference)

   NOTE2: There's some code below that you can copy to a simple
   program to test the tree container ... works jest fine ...
   --------------------------------------------------------------
*/

CLASS kmTreeContainer(parentObj) of KMCUSTCONTAINER(parentObj) custom
   with (this)
      width  := 18.4286
      height := 10.1364
      metric := 0	// Chars
   endwith

   this.KMTREEVIEW1 = new KMTREEVIEW(this)
   with (this.KMTREEVIEW1)
      height := 7.9545
      left   := 1.4286
      top    := 1.5
      width  := 15
   endwith

   this.EXPANDPUSHBUTTON = new PUSHBUTTON(this)
   with (this.EXPANDPUSHBUTTON)
      height   := 0.8636
      left     := 4.5714
      top      := 0.4545
      width    := 3.8571
      text     := ""
      speedBar := true
      upBitmap := "RESOURCE TS_APPEND"
      speedTip := "Expand tree items"
      onClick  := {|| this.parent.kmTreeView1.expand() }
   endwith

   this.COLLAPSEPUSHBUTTON = new PUSHBUTTON(this)
   with (this.COLLAPSEPUSHBUTTON)
      height   := 0.8636
      left     := 8.7143
      top      := 0.4545
      width    := 3.8571
      text     := ""
      speedBar := true
      upBitmap := "RESOURCE TS_DELETE"
      speedTip := "Collapse tree items"
      onClick  := {||this.parent.kmTreeView1.collapse() }
   endwith

   function FillIt // we're really just calling the super class's
                   // TreeFill method ...
      parameter aArray 
      this.kmTreeView1.TreeFill( this.kmTreeView1, aArray )
endclass

/*
   ----------------------------------------------------------------
   The following code can be copied and pasted to test
   the treeview container on a form ...:

   // copy from here -----------------------
   set proc to formcntl.cc additive
   // create array:
   a = {"item1",{"item11","item12"},;
        "item2",;
        "item3",;
        "item4",{"item41",;
                 "item42",{"item42_1",;
                           "item42_2",;
                           "item42_3"}},;
        "item5"}

   // instantiate form:
   f = new form()
   f.text = "Test kmTreeContainer"
   f.autoCenter = true
   // instantiate container:
   f.t = new kmTreeContainer( f )
   // load treeview object with items in array 'a'
   f.t.fillit( a)
   // open form
   f.open()
   // to here ------------------------------
   ----------------------------------------------------------------
*/
// end of sample code for kmtreeContainer object

function Repaint
/*
   ----------------------------------------------------------------
   This is a routine by Jim Sare with minor
   modifications by Ken Mayer. It can be used
   to ensure that if any processing has occurred
   that the form gets repainted (which sometimes
   doesn't happen) ...
   ----------------------------------------------------------------
*/
   parameter oForm
   if type( "oForm" ) == "U" or empty( oForm )
      oForm = form
   endif
   if type( "UpdateWindow" ) # "FP"
      extern CLOGICAL UpdateWindow( CHANDLE ) USER32
   endif
RETURN UpdateWindow( oForm.hWnd )
// end of function: Repaint()


/*
   ----------------------------------------------------------------
   This form is by Gary White and is used with
   the kmCustDateEntryField2 above -- it is a
   simple modal (child) form used to ask if the 
   user wants to move the date forward/backward
   off the holiday/weekend found.
   I set the onOpen event for the form so we can force
   the "nextWorkDay" button to be the one with focus 
   (pressing "Enter" will cause this button to be the
   one to fire, unless user changes buttons)
   ----------------------------------------------------------------
*/
class VerifyDateForm of FORM
   with (this)
      onOpen        := {; this.nextWorkDay.setFocus() }
   	mdi           := false
      scaleFontBold := false
      height        := 6.1364
      left          := 39.1429
      top           := 6.4545
      width         := 43.2857
      text          := "Verify Date"
      smallTitle    := true
      autoCenter    := true
      moveable      := true
   endwith

   this.TEXT1 = new TEXT(this)
   with (this.TEXT1)
      height = 3.8
      left = 0
      top = 0
      width = 43.2857
      border = true
      anchor = 2	// Top
      marginHorizontal = 1
      marginVertical = 0.2
      text = "The date you entered is either a weekend, or a holiday.  You may move this date to the first working day prior to that date, the first working day after that date, or leave it as is."
      borderStyle = CONTAINERBORDER // defined at top of this file
   endwith

   this.PRIORWORKDAY = new PUSHBUTTON(this)
   with (this.PRIORWORKDAY)
      height = 1.0909
      left = 1.2857
      top = 4.5
      width = 13.4286
      text = "Prior Workday"
      value = false
      onclick = {;form.parent.transfer="B";form.close()}
   endwith

   this.NEXTWORKDAY = new PUSHBUTTON(this)
   with (this.NEXTWORKDAY)
      height = 1.0909
      left = 14.7143
      top = 4.5
      width = 13.4286
      text = "Next Workday"
      value = false
      onclick = {;form.parent.transfer="F";form.close()}
   endwith

   this.LEAVEIT = new PUSHBUTTON(this)
   with (this.LEAVEIT)
      height = 1.0909
      left = 28.1429
      top = 4.5
      width = 13.4286
      text = "Leave As Is"
      value = false
      onclick = {;form.parent.transfer="X";form.close()}
   endwith
endclass
// End of Form: VerifyDateForm

#include <duflp.h>

/*
   ----------------------------------------------------------------
   End of file: FORMCNTL.CC
   ----------------------------------------------------------------
*/



